/** TODO PORT ME TO <eg-combobox> */
import {Component, OnInit, Input, Output, ViewChild, EventEmitter, AfterViewInit} from '@angular/core';
import {Observable, Subject, map, mapTo, debounceTime, distinctUntilChanged, mergeWith as merge, filter} from 'rxjs';
import {AuthService} from '@eg/core/auth.service';
import {ServerStoreService} from '@eg/core/server-store.service';
import {OrgService} from '@eg/core/org.service';
import {IdlObject} from '@eg/core/idl.service';
import {PermService} from '@eg/core/perm.service';
import {NgbTypeahead, NgbTypeaheadSelectItemEvent} from '@ng-bootstrap/ng-bootstrap';
import {FormControl, FormGroup} from '@angular/forms';

/** Org unit selector
 *
 * The following precedence is used when applying a load-time value
 *
 * 1. initialOrg / initialOrgId
 * 2. Value from server setting specificed with persistKey (fires onload).
 * 3. Value from fallbackOrg / fallbackOrgId (fires onload).
 * 4. Default applyed when applyDefault is set (fires onload).
 *
 * Users can detect when the component has completed its load-time
 * machinations by subscribing to the componentLoaded Output which
 * fires exactly once when loading is completed.
 */

// Use a unicode char for spacing instead of ASCII=32 so the browser
// won't collapse the nested display entries down to a single space.
const PAD_SPACE = ' '; // U+2007

interface OrgDisplay {
  id: number;
  label: string;
  disabled: boolean;
}

@Component({
    selector: 'eg-org-select',
    templateUrl: './org-select.component.html'
})
export class OrgSelectComponent implements OnInit, AfterViewInit {
    static _domId = 0;

    showCombinedNames = false; // Managed via user/workstation setting

    _selected: OrgDisplay;
    set selected(s: OrgDisplay) {
        if (s !== this._selected) {
            this._selected = s;

            // orgChanged() does not fire when the value is cleared,
            // so emit the onChange here for cleared values only.
            if (!s) { // may be '' or null
                this._selected = null;
                this.onChange.emit(null);
            }
        }
    }

    get selected(): OrgDisplay {
        return this._selected;
    }

    click$ = new Subject<string>();
    valueFromSetting: number = null;
    sortedOrgs: IdlObject[] = [];
    orgSelectGroup: FormGroup;
    controller: HTMLInputElement;

    // Disable the entire input
    @Input() disabled: boolean;

    @ViewChild('instance', { static: false }) instance: NgbTypeahead;

    // Placeholder text for selector input
    @Input() placeholder = '';

    // ARIA label for selector. Required if there is no <label> in the markup.
    @Input() ariaLabel?: string;

    // Optionally provide an aria-labelledby for the input.  This should be one or more
    // space-delimited ids of elements that describe this combobox.
    @Input() ariaLabelledby: string;

    // ARIA describedby, for attaching error messages
    @Input() ariaDescribedby?: string = null;

    // ID to display in the DOM for this selector
    @Input() domId = 'eg-org-select-' + OrgSelectComponent._domId++;

    // String containing additional CSS class names, space separated
    @Input() moreClasses = '';

    @Input() name = '';

    // Org unit field displayed in the selector
    @Input() displayField = 'shortname';

    // if no initialOrg is provided, none could be found via persist
    // setting, and no fallbackoOrg is provided, apply a sane default.
    // First tries workstation org unit, then user home org unit.
    // An onChange event WILL be generated when a default is applied.
    @Input() applyDefault = false;

    @Input() readOnly = false;

    @Input() required = false;

    @Input() ngbAutofocus = null; // passthrough for [ngbAutofocus]

    // List of org unit IDs to exclude from the selector
    hidden: number[] = [];
    @Input() set hideOrgs(ids: number[]) {
        if (ids) { this.hidden = ids; }
    }

    // List of org unit IDs to disable in the selector
    _disabledOrgs: number[] = [];
    @Input() set disableOrgs(ids: number[]) {
        if (ids) { this._disabledOrgs = ids; }
    }

    get disableOrgs(): number[] {
        return this._disabledOrgs;
    }

    // Apply an org unit value at load time.
    // These will NOT result in an onChange event.
    @Input() initialOrg: IdlObject;
    @Input() initialOrgId: number;

    // Value is persisted via server setting with this key.
    // Key is prepended with 'eg.orgselect.'
    @Input() persistKey: string;

    // If no initialOrg is provided and no value could be found
    // from a persist setting, fall back to one of these values.
    // These WILL result in an onChange event
    @Input() fallbackOrg: IdlObject;
    @Input() fallbackOrgId: number;

    // Modify the selected org unit via data binding.
    // This WILL NOT result in an onChange event firing.
    @Input() set applyOrg(org: IdlObject) {
        this.selected = org ? this.formatForDisplay(org) : null;
        this.updateValidity(this.selectedOrgId());
    }

    // Modify the selected org unit by ID via data binding.
    // This WILL NOT result in an onChange event firing.
    @Input() set applyOrgId(id: number) {
        this.selected = id ? this.formatForDisplay(this.org.get(id)) : null;
        this.updateValidity(this.selectedOrgId());
    }

    // Limit org unit display to those where the logged in user
    // has the following permissions.
    permLimitOrgs: number[];
    @Input() set limitPerms(perms: string[]) {
        this.applyPermLimitOrgs(perms);
    }

    // Function which should return a string value representing
    // a CSS class name to use for styling each org unit label
    // in the selector.
    @Input() orgClassCallback: (orgId: number) => string;

    // Emitted when the Enter key is pressed in the input and the popup is not open
    @Output() orgSelectEnter = new EventEmitter<number>();

    // Emitted when a key is pressed in the input and the popup is not open.
    // A passthrough for keyboard events on the input.
    // Example: (orgSelectKey)="$event.key === 'Escape' ? cancel() : handleKeydown($event)"
    @Output() orgSelectKey = new EventEmitter<Event>();

    // Emitted when the org unit value is changed via the selector.
    // Does not fire on initialOrg
    // eslint-disable-next-line @angular-eslint/no-output-on-prefix
    @Output() onChange = new EventEmitter<IdlObject>();

    // Emitted once when the component is done fetching settings
    // and applying its initial value.  For apps that use the value
    // of this selector to load data, this event can be used to reliably
    // detect when the selector is done with all of its automated
    // underground shuffling and landed on a value.
    @Output() componentLoaded: EventEmitter<void> = new EventEmitter<void>();

    // convenience method to get an IdlObject representing the current
    // selected org unit. One way of invoking this is via a template
    // reference variable.
    selectedOrg(): IdlObject {
        // eslint-disable-next-line eqeqeq
        if (this.selected == null) {
            return null;
        }
        return this.org.get(this.selected.id);
    }

    selectedOrgId(): number {
        return this.selected ? this.selected.id : null;
    }

    constructor(
      private auth: AuthService,
      private serverStore: ServerStoreService,
      private org: OrgService,
      private perm: PermService,
    ) {
        this.orgClassCallback = (orgId: number): string => '';
        this.orgSelectGroup = new FormGroup({
            orgSelect: new FormControl()
        });
    }

    ngOnInit() {


        let promise = this.persistKey ?
            this.getFromSetting() : Promise.resolve(null);

        promise = promise.then(startupOrg => {
            return this.serverStore.getItem('eg.orgselect.show_combined_names')
                .then(show => {
                    const sortField = show ? 'name' : this.displayField;

                    // Sort the tree and reabsorb to propagate the sorted
                    // nodes to the org.list() used by this component.
                    // Maintain our own copy of the org list in case the
                    // org service is sorted in a different manner by other
                    // parts of the code.
                    this.org.sortTree(sortField);
                    this.org.absorbTree();
                    this.sortedOrgs = this.org.list();

                    this.showCombinedNames = show;
                })
                .then(_ => startupOrg);
        });

        promise.then((startupOrgId: number) => {

            if (!startupOrgId) {

                if (this.selected) {
                    // A value may have been applied while we were
                    // talking to the network.
                    startupOrgId = this.selected.id;

                } else if (this.initialOrg) {
                    startupOrgId = this.initialOrg.id();

                } else if (this.initialOrgId) {
                    startupOrgId = this.initialOrgId;

                } else if (this.fallbackOrgId) {
                    startupOrgId = this.fallbackOrgId;

                } else if (this.fallbackOrg) {
                    startupOrgId = this.org.get(this.fallbackOrg).id();

                } else if (this.applyDefault && this.auth.user()) {
                    startupOrgId = this.auth.user().ws_ou();
                }
            }

            let startupOrg;
            if (startupOrgId) {
                startupOrg = this.org.get(startupOrgId);
                this.selected = this.formatForDisplay(startupOrg);
            }

            this.markAsLoaded(startupOrg);
        });
    }

    ngAfterViewInit(): void {
        document.querySelectorAll('ngb-typeahead-window button[disabled]').forEach(b => b.setAttribute('tabindex', '-1'));
        this.controller = this.instance['_nativeElement'] as HTMLInputElement;
        this.controller.addEventListener('keydown', this.onKeydown.bind(this));
    }

    getDisplayLabel(org: IdlObject): string {
        if (this.showCombinedNames) {
            return `${org.name()} (${org.shortname()})`;
        } else {
            return org[this.displayField]();
        }
    }

    getFromSetting(): Promise<number> {

        const key = `eg.orgselect.${this.persistKey}`;

        return this.serverStore.getItem(key).then(
            value => this.valueFromSetting = value
        );
    }

    // Indicate all load-time shuffling has completed.
    markAsLoaded(onChangeOrg?: IdlObject) {
        setTimeout(() => { // Avoid emitting mid-digest
            this.componentLoaded.emit();
            this.componentLoaded.complete();
            if (onChangeOrg) { this.onChange.emit(onChangeOrg); }
        });
    }

    //
    applyPermLimitOrgs(perms: string[]) {

        if (!perms) {
            return;
        }

        // handle lazy clients that pass null perm names
        perms = perms.filter(p => p !== null && p !== undefined);

        if (perms.length === 0) {
            return;
        }

        // NOTE: If permLimitOrgs is useful in a non-staff context
        // we need to change this to support non-staff perm checks.
        this.perm.hasWorkPermAt(perms, true).then(permMap => {
            this.permLimitOrgs =
                // safari-friendly version of Array.flat()
                Object.values(permMap).reduce((acc, val) => acc.concat(val), []);
        });
    }

    // Format for display in the selector drop-down and input.
    formatForDisplay(org: IdlObject): OrgDisplay {
        let label = this.getDisplayLabel(org);
        if (!this.readOnly) {
            label = PAD_SPACE.repeat(org.ou_type().depth()) + label;
        }
        return {
            id : org.id(),
            label : label,
            disabled : this.disableOrgs.includes(org.id())
        };
    }

    // Fired by the typeahead to inform us of a change.
    orgChanged(selEvent: NgbTypeaheadSelectItemEvent) {
        // console.debug('org unit change occurred ' + selEvent.item);
        const newOrg = selEvent.item.id;
        this.onChange.emit(this.org.get(newOrg));

        if (this.persistKey && this.valueFromSetting !== selEvent.item.id) {
            // persistKey is active.  Update the persisted value when changed.

            const key = `eg.orgselect.${this.persistKey}`;
            this.valueFromSetting = selEvent.item.id;
            this.serverStore.setItem(key, this.valueFromSetting);
        }
    }

    updateValidity(newOrg: number) {
        if (newOrg && this.required) {
            // console.debug('Checking org validity via FormControl', this.orgSelectGroup.controls.orgSelect);
            if (this.isValidOrg(newOrg)) {
                return this.orgSelectGroup.controls.orgSelect.valid;
            }
            return this.orgSelectGroup.controls.orgSelect.invalid;
        }
    }

    isValidOrg(org: any) : boolean {
        if (!org) { return false; }

        if (this.disableOrgs.includes(org)) { return false; }

        return true;
    }

    // Remove the tree-padding spaces when matching.
    formatter = (result: OrgDisplay) => result ? result.label.trim() : '';

    // reset the state of the component
    reset() {
        this.selected = null;
    }

    onKeydown($event: KeyboardEvent) {
        // console.debug('Key: ', $event);

        if (this.instance.isPopupOpen()) {
            if ($event.key === 'ArrowUp' || $event.key === 'ArrowDown') {
                this.scrollEntries();
            }
            return;
        }

        if ( $event.key === 'ArrowDown' && $event.ctrlKey && $event.shiftKey ) {
            setTimeout(() => this.openMe($event));
            return;
        }

        // a shortcut to the Org ID if Enter is the only key event you're interested in
        if ( $event.key === 'Enter' ) {
            this.onEnter();
        }

        // Pass through to calling component via (orgSelectKey)
        this.orgSelectKey.emit($event);
    }

    onEnter() {
        this.orgSelectEnter.emit(this.selected.id);
    }

    openMe($event) {
        // Give the input a chance to focus then fire the click
        // handler to force open the typeahead
        document.getElementById(this.domId).focus();
        setTimeout(() => this.click$.next(''));
    }

    closeMe($event) {
        this.instance.dismissPopup();
    }

    scrollEntries() {
        // adapted from https://github.com/ng-bootstrap/ng-bootstrap/issues/4789
        if (!this.controller) {return;}

        const listbox = document.getElementById(this.controller.getAttribute('aria-owns'));
        // console.debug("Listbox: ", listbox);

        const activeItem = document.getElementById(this.controller.getAttribute('aria-activedescendant'));
        if (activeItem) {
            if (activeItem.offsetTop < listbox.scrollTop) {
                listbox.scrollTo({ top: activeItem.offsetTop });
            } else if (activeItem.offsetTop + activeItem.offsetHeight > listbox.scrollTop + listbox.clientHeight) {
                listbox.scrollTo({ top: activeItem.offsetTop + activeItem.offsetHeight - listbox.clientHeight });
            }
        }
    }

    // NgbTypeahead doesn't offer a way to style the dropdown
    // button directly, so we have to reach up and style it ourselves.
    applyDisableStyle() {
        this.disableOrgs.forEach(id => {
            const node = document.getElementById(`${this.domId}-${id}`);
            if (node) {
                const button = node.parentNode as HTMLElement;
                button.classList.add('disabled');
            }
        });
    }

    // Free-text values are not allowed.
    handleBlur() {
        if (typeof this.selected === 'string') {
            this.selected = null;
        }
    }

    filter = (text$: Observable<string>): Observable<OrgDisplay[]> => {

        return text$.pipe(
            // eslint-disable-next-line no-magic-numbers
            debounceTime(200),
            distinctUntilChanged(),
            merge(
                // Inject a specifier indicating the source of the
                // action is a user click
                this.click$.pipe(filter(() => !this.instance.isPopupOpen()))
                    .pipe(mapTo('_CLICK_'))
            ),
            map(term => {

                let orgs = this.sortedOrgs.filter(org =>
                    this.hidden.filter(id => org.id() === id).length === 0
                );

                if (this.permLimitOrgs) {
                    // Avoid showing org units where the user does
                    // not have the requested permission.
                    orgs = orgs.filter(org =>
                        this.permLimitOrgs.includes(org.id()));
                }

                if (term !== '_CLICK_') {
                    // For search-driven events, limit to the matching
                    // org units.
                    orgs = orgs.filter(org => {
                        return term === '' || // show all
                            this.getDisplayLabel(org)
                                .toLowerCase().indexOf(term.toLowerCase()) > -1;

                    });
                }

                // Give the typeahead a chance to open before applying
                // the disabled org unit styling.
                setTimeout(() => this.applyDisableStyle());

                return orgs.map(org => this.formatForDisplay(org));
            })
        );
    };
}


