import {Component, Input, Output, OnInit, AfterViewInit, EventEmitter,
    OnDestroy, ChangeDetectorRef, ViewChild, ViewEncapsulation} from '@angular/core';
import {IdlService} from '@eg/core/idl.service';
import {OrgService} from '@eg/core/org.service';
import {ServerStoreService} from '@eg/core/server-store.service';
import {FormatService} from '@eg/core/format.service';
import {GridContext, GridColumn, GridDataSource,
    GridCellTextGenerator, GridRowFlairEntry} from './grid';
import {GridToolbarComponent} from './grid-toolbar.component';

/**
 * Main grid entry point.
 */

@Component({
    selector: 'eg-grid',
    templateUrl: './grid.component.html',
    styleUrls: ['grid.component.css'],
    // share grid css globally once imported so all grid component CSS
    // can live in grid.component.css and to avoid multiple copies of
    // the CSS when multiple grids are displayed.
    encapsulation: ViewEncapsulation.None
})

export class GridComponent implements OnInit, AfterViewInit, OnDestroy {

    // Source of row data.
    @Input() dataSource: GridDataSource;

    // IDL class for auto-generation of columns
    @Input() idlClass: string;

    // Comma-separated list of column names, to set the order of columns
    // from auto-generated columns only
    @Input() autoGeneratedColumnOrder: string;

    // True if any columns are sortable
    @Input() sortable: boolean;

    // True if the grid supports sorting of multiple columns at once
    @Input() multiSortable: boolean;

    // If true, grid sort requests only operate on data that
    // already exists in the grid data source -- no row fetching.
    // The assumption is all data is already available.
    @Input() useLocalSort: boolean;

    // Storage persist key / per-grid-type unique identifier
    // The value is prefixed with 'eg.grid.'
    //
    // If persistKey is set to "disabled", or does not exist,
    // the grid will not display a Save button to the user
    @Input() persistKey: string;

    @Input() disableSelect: boolean;

    // Prevent selection of multiple rows
    @Input() disableMultiSelect: boolean;

    // Show an extra column in the grid where the caller can apply
    // row-specific flair (material icons).
    @Input() rowFlairIsEnabled: boolean;

    // Title (visible only in tooltip) for the flair column header
    @Input() flairColumnHeader: string = $localize `Notifications`;

    // Returns a material icon name to display in the flair column
    // (if enabled) for the given row.
    @Input() rowFlairCallback: (row: any) => GridRowFlairEntry;

    // Returns a space-separated list of CSS class names to apply to
    // a given row
    @Input() rowClassCallback: (row: any) => string;

    // Returns a space-separated list of CSS class names to apply to
    // a given cell or all cells in a column.
    @Input() cellClassCallback: (row: any, col: GridColumn) => string;

    // comma-separated list of fields to show by default.
    // This field takes precedence over hideFields.
    // When a value is applied, any field not in this list will
    // be hidden.
    @Input() showFields: string;

    // comma-separated list of fields to hide.
    // This does not imply all other fields should be visible, only that
    // the selected fields will be hidden.
    @Input() hideFields: string;

    // comma-separated list of fields to ignore when generating columns
    // from the IDL.
    // This does not imply all other fields should be available, only
    // that the selected fields will be ignored.
    @Input() ignoreFields: string;

    // When true, only display columns that are declared in the markup
    // and leave all auto-generated fields hidden.
    @Input() showDeclaredFieldsOnly: boolean;

    // Allow the caller to jump directly to a specific page of
    // grid data.
    @Input() pageOffset: number;
    // Pass in a default page size.  May be overridden by settings.
    @Input() pageSize: number;

    @Input() showLinkSelectors: boolean;

    @Input() disablePaging: boolean;

    // result filtering
    //
    // filterable: true if the result filtering controls
    // should be displayed
    @Input() filterable: boolean;

    // initialFilterValues: a hash of values you want the
    // filters to start off with
    @Input() initialFilterValues: {[field: string]: string};

    // allowNamedFilterSets: true if the result filtering
    // controls can be saved or loaded via a name
    @Input() allowNamedFilterSets: boolean;

    // migrateLegacyFilterSets: if set to a legacy filter interface type
    // (i.e. url_verify), attempt to migrate any legacy filter sets for
    // that interface.
    @Input() migrateLegacyFilterSets: string;

    // sticky grid header
    //
    // stickyHeader: true of the grid header should be
    // "sticky", i.e., remain visible if if the table is long
    // and the user has scrolled far enough that the header
    // would go out of view
    @Input() stickyHeader: boolean;

    @Input() cellTextGenerator: GridCellTextGenerator;

    // If set, appears along the top left side of the grid.
    @Input() toolbarLabel: string;

    // If true, showing/hiding columns will force the data source to
    // refresh the current page of data.
    @Input() reloadOnColumnChange = false;

    context: GridContext;
    private static id = 0;
    gridDomId = 'eg-grid-';
    grid_density = 'normal';

    // These events are emitted from our grid-body component.
    // They are defined here for ease of access to the caller.
    // eslint-disable-next-line @angular-eslint/no-output-on-prefix
    @Output() onRowActivate: EventEmitter<any>;
    // eslint-disable-next-line @angular-eslint/no-output-on-prefix
    @Output() onRowClick: EventEmitter<any>;

    // Emits an array of grid row indexes on any row selection change.
    @Output() rowSelectionChange: EventEmitter<string[]>;

    @ViewChild('toolbar', { static: true }) toolbar: GridToolbarComponent;

    constructor(
        private idl: IdlService,
        private org: OrgService,
        private store: ServerStoreService,
        private format: FormatService,
        private cdr: ChangeDetectorRef
    ) {
        this.context =
            new GridContext(this.idl, this.org, this.store, this.format, this.cdr);
        this.onRowActivate = new EventEmitter<any>();
        this.onRowClick = new EventEmitter<any>();
        this.rowSelectionChange = new EventEmitter<string[]>();
    }

    ngOnInit() {

        if (!this.dataSource) {
            throw new Error('[egGrid] requires a [dataSource]');
        }

        this.gridDomId += GridComponent.id++;
        this.context.gridDomId = this.gridDomId;
        this.context.idlClass = this.idlClass;
        this.context.dataSource = this.dataSource;
        this.context.persistKey = this.persistKey;
        this.context.autoGeneratedColumnOrder = this.autoGeneratedColumnOrder;
        this.context.isSortable = this.sortable === true;
        this.context.isFilterable = this.filterable === true;
        this.context.initialFilterValues = this.initialFilterValues || null;
        this.context.allowNamedFilterSets = this.allowNamedFilterSets === true;
        this.context.migrateLegacyFilterSets = this.migrateLegacyFilterSets;
        this.context.stickyGridHeader = this.stickyHeader === true;
        this.context.isMultiSortable = this.multiSortable === true;
        this.context.useLocalSort = this.useLocalSort === true;
        this.context.disableSelect = this.disableSelect === true;
        this.context.disableMultiSelect = this.disableMultiSelect === true;
        this.context.rowFlairIsEnabled = this.rowFlairIsEnabled  === true;
        this.context.showDeclaredFieldsOnly = this.showDeclaredFieldsOnly;
        this.context.rowFlairCallback = this.rowFlairCallback;
        this.context.flairColumnHeader = this.flairColumnHeader;
        this.context.toolbarLabel = this.toolbarLabel;
        this.context.disablePaging = this.disablePaging === true;
        this.context.cellTextGenerator = this.cellTextGenerator;
        this.context.ignoredFields = [];
        this.context.reloadOnColumnChange = this.reloadOnColumnChange;

        if (this.showFields) {
            // Stripping spaces allows users to add newlines to
            // long lists of field names without consequence.
            this.showFields = this.showFields.replace(/\s+/g, '');
            this.context.defaultVisibleFields = this.showFields.split(',');
        }
        if (this.hideFields) {
            this.hideFields = this.hideFields.replace(/\s+/g, '');
            this.context.defaultHiddenFields = this.hideFields.split(',');
        }
        if (this.ignoreFields) {
            this.ignoreFields = this.ignoreFields.replace(/\s+/g, '');
            this.context.ignoredFields = this.ignoreFields.split(',');
        }

        if (this.pageOffset) {
            this.context.pager.offset = this.pageOffset;
        }

        if (this.pageSize) {
            this.context.pager.limit = this.pageSize;
        }

        // TS doesn't seem to like: let foo = bar || () => '';
        this.context.rowClassCallback =
            this.rowClassCallback || function () { return ''; };
        this.context.cellClassCallback =
            this.cellClassCallback || function () { return ''; };

        this.context.rowSelector.selectionChange.subscribe(
            keys => this.rowSelectionChange.emit(keys)
        );

        if (this.showLinkSelectors) {
            console.debug(
                'showLinkSelectors is deprecated and no longer has any effect');
        }

        this.context.init();
    }

    ngAfterViewInit() {
        this.context.initData();
        this.grid_density = this.context.grid_density;
    }

    ngOnDestroy() {
        // eslint-disable-next-line rxjs-x/no-subject-unsubscribe
        this.context.rowSelector.selectionChange.unsubscribe();
        this.context.destroy();
    }

    print = () => {
        this.toolbar.printHtml();
    };

    reload() {
        this.context.reload();
    }
    reloadWithoutPagerReset() {
        this.context.reloadWithoutPagerReset();
    }

    // Not using @HostListener because it only works globally.
    onGridKeyDown(evt: KeyboardEvent) {
        switch (evt.key) {
            case 'ArrowUp':
                if (evt.shiftKey) {
                    // Extend selection up one row
                    this.context.selectMultiRowsPrevious();
                } else {
                    this.context.selectPreviousRow();
                }
                evt.stopPropagation();
                break;
            case 'ArrowDown':
                if (evt.shiftKey) {
                    // Extend selection down one row
                    this.context.selectMultiRowsNext();
                } else {
                    this.context.selectNextRow();
                }
                evt.stopPropagation();
                break;
            case 'ArrowLeft':
                // skip if the key has been pressed on a column resize handle
                // see keyboard events in grid-header component
                if (!this.context.currentResizeCol) {
                    this.context.toPrevPage()
                        .then(ok => this.context.selectFirstRow(), err => {});
                    evt.stopPropagation();
                }
                break;
            case 'PageUp':
                this.context.toPrevPage()
                    .then(ok => this.context.selectFirstRow(), err => {});
                evt.stopPropagation();
                break;
            case 'ArrowRight':
                // skip if the key has been pressed on a column resize handle
                if (!this.context.currentResizeCol) {
                    this.context.toNextPage()
                        .then(ok => this.context.selectFirstRow(), err => {});
                    evt.stopPropagation();
                }
                break;
            case 'PageDown':
                this.context.toNextPage()
                    .then(ok => this.context.selectFirstRow(), err => {});
                evt.stopPropagation();
                break;
            case 'a':
                // control-a means select all visible rows.
                // For consistency, select all rows in the current page only.
                if (evt.ctrlKey) {
                    this.context.rowSelector.clear();
                    this.context.selectRowsInPage();
                    evt.preventDefault();
                }
                break;

            case 'Enter':
                if (this.context.lastSelectedIndex) {
                    this.onRowActivate.emit(
                        this.context.getRowByIndex(
                            this.context.lastSelectedIndex)
                    );
                }
                evt.stopPropagation();
                break;
        }
    }

}



