import type { InvalidTestCase, SuggestionOutput, TestLanguageOptions } from '@typescript-eslint/rule-tester';
/**
 * When leveraging the convertAnnotatedSourceToFailureCase() utility, the
 * following characters are eligible to be used in the source code of expected
 * failure cases within ESLint unit tests in order to provide an easy way to
 * annotate where one or more ESLint errors are expected to occur within that
 * source.
 *
 * See the convertAnnotatedSourceToFailureCase() utility itself for more details.
 */
export declare const SPECIAL_UNDERLINE_CHARS: readonly ["~", "^", "#", "%", "¶", "*", "¨", "@"];
type MultipleErrorOptions<TMessageIds extends string, Options extends readonly unknown[]> = BaseErrorOptions<Options> & {
    readonly languageOptions?: TestLanguageOptions;
    readonly messages: readonly (Message<TMessageIds> & {
        readonly char: (typeof SPECIAL_UNDERLINE_CHARS)[number];
    })[];
};
type BaseErrorOptions<Options> = {
    readonly description: string;
    readonly annotatedSource: string;
    readonly options?: Options;
    readonly annotatedOutput?: string;
    readonly annotatedOutputs?: readonly string[];
    readonly filename?: string;
    readonly only?: boolean;
};
type Message<TMessageIds extends string> = {
    readonly messageId: TMessageIds;
    readonly data?: Record<string, unknown>;
    readonly suggestions?: SuggestionOutput<TMessageIds>[];
};
type SingleErrorOptions<TMessageIds extends string, Options extends readonly unknown[]> = BaseErrorOptions<Options> & Message<TMessageIds> & {
    readonly languageOptions?: TestLanguageOptions;
};
/**
 * convertAnnotatedSourceToFailureCase() provides an ergonomic way to easily write
 * expected failure cases for ESLint rules by allowing you to directly annotate the
 * source code for the case with one or more of the values in `SPECIAL_UNDERLINE_CHARS`.
 *
 * This not only makes the unit tests easier to write because of the time saved in figuring
 * out location data in terms of lines and columns, but also far easier to read, which is
 * arguably much more important.
 *
 * Here is a real-world example of using the utility:
 *
 * ```ts
 *  convertAnnotatedSourceToFailureCase({
 *    description: 'should fail when Pipe has no prefix ng',
 *    annotatedSource: `
 *        @Pipe({
 *          name: 'foo-bar'
 *                ~~~~~~~~~
 *        })
 *        class Test {}
 *    `,
 *    messageId: 'pipePrefix,
 *    options: [{ prefixes: ['ng'] }],
 *    data: { prefixes: '"ng"' },
 *  }),
 * ```
 */
export declare function convertAnnotatedSourceToFailureCase<TMessageIds extends string, Options extends readonly unknown[]>(errorOptions: SingleErrorOptions<TMessageIds, Options>): InvalidTestCase<TMessageIds, Options>;
export declare function convertAnnotatedSourceToFailureCase<TMessageIds extends string, Options extends readonly unknown[]>(errorOptions: MultipleErrorOptions<TMessageIds, Options>): InvalidTestCase<TMessageIds, Options>;
export {};
//# sourceMappingURL=convert-annotated-source-to-failure-case.d.ts.map