#!/usr/bin/env perl

use strict;
use warnings;
use Test::More;
use LWP::UserAgent;
use HTTP::Cookies;
use HTTP::Request::Common;
use JSON;
use Digest::MD5 qw(md5_hex);
use Time::HiRes qw(time);
use DateTime;
use DateTime::Format::ISO8601;
use Data::Dumper;

# Configuration
my $host = $ENV{TEST_HOST} || 'localhost';
my $SHARED_SECRET = 'your_shared_secret_here';
my $VENDOR_USERNAME = 'admin';
#  TODO: add this to concerto: SELECT actor.set_passwd(1, 'ecard_vendor', 'ecard_password');
#  TODO: insert into actor.org_unit_setting (name, org_unit, value) values
#           ('vendor.quipu.ecard.shared_secret', 1, '"your_shared_secret_here"'),
#           ('vendor.quipu.ecard.admin_org_unit', 1, 4),
#           ('vendor.quipu.ecard.barcode_length', 4, 11),
#           ('vendor.quipu.ecard.barcode_prefix', 4, '"321"'),
#           ('vendor.quipu.ecard.calculate_checkdigit', 4, 'true'),
#           ('vendor.quipu.ecard.admin_usrname', 4, '"admin"'),
#           ('vendor.quipu.ecard.patron_profile', 4, 2),
#           ('vendor.quipu.ecard.account_id', 4, 1234),
#           ('opac.ecard_registration_enabled', 1, true),
#           ('opac.ecard_renewal_enabled', 1, true);
# TODO: insert into actor.stat_cat (owner, name, opac_visible, usr_summary, checkout_archive, required, allow_freetext)
#       values (1, 'Nintendo or Sega?', true, true, false, false, false);
#       insert into actor.stat_cat_entry (stat_cat, owner, value)
#       values (1, 1, 'Nintendo'), (1, 1, 'Sega');
my $VENDOR_PASSWORD = 'ecard_password';
my $ECARD_SUBMIT_URL = "https://$host/eg/opac/ecard/submit";
my $OSRF_GATEWAY_URL = "https://$host/osrf-gateway-v1";

# Test data
my %ecard_data = (
    first_given_name => 'John',
    second_given_name => 'Michael',
    family_name => 'Doe',
    email => 'johndoe@example.com',
    dob => '1990-01-15',
    day_phone => '555-123-4567',
    home_ou => '4',
    passwd => 'demo123',
    ident_type => '1',
    ident_value => 'DL12345678',
    physical_street1 => '123 Main St',
    physical_city => 'Anytown',
    physical_state => 'Anystate',
    physical_country => 'USA',
    physical_county => 'Anycounty',
    physical_post_code => '12345',
    mailing_street1 => '123 Main St',
    mailing_city => 'Anytown',
    mailing_state => 'Anystate',
    mailing_country => 'USA',
    mailing_county => 'Anycounty',
    mailing_post_code => '12345',
);

# Helper Functions
sub generate_ecard_token {
    my ($shared_secret, $custom_timestamp) = @_;
    my $timestamp = defined($custom_timestamp) ? $custom_timestamp : int(time());
    my $signature = md5_hex($timestamp . $shared_secret);
    return "$timestamp:$signature";
}

sub create_ua {
    return LWP::UserAgent->new(
        ssl_opts => {
            verify_hostname => 0,
            SSL_verify_mode => 0x00
        }
    );
}

sub ecard_submit_request {
    my ($ua, $params) = @_;
    my $request = POST $ECARD_SUBMIT_URL, $params;
    return $ua->request($request);
}

sub osrf_gateway_request {
    my ($ua, $params) = @_;
    my $request = POST $OSRF_GATEWAY_URL, $params;
    return $ua->request($request);
}

# Begin Tests

subtest 'API Tests' => sub {
    my $ua = create_ua();

    # https://host/osrf-gateway-v1?service=open-ils.auth&method=open-ils.auth.login&param=

    my $auth_token;

    # Test: auth token
    {
        my $response = osrf_gateway_request($ua, [
            service => 'open-ils.auth',
            method => 'open-ils.auth.login',
            param => encode_json({ username => "admin", password => "demo123" }) # for expediency; the vendor credentials we added earlier will not suffice here currently
        ]);
        ok($response->is_success, 'Login submission succeeds');
        my $content = eval { decode_json($response->decoded_content) };
        if (defined $content->{payload} && defined $content->{payload}->[0] && defined $content->{payload}->[0]->{payload} && defined $content->{payload}->[0]->{payload}->{authtoken}) {
            $auth_token = $content->{payload}->[0]->{payload}->{authtoken};
        }
        ok($auth_token, 'Received authtoken');
        diag(Dumper($auth_token || $content));
    }

    my $first_patron_id;

    # Test: Valid submission
    {
        my $response = ecard_submit_request($ua, {
            %ecard_data,
            asceum => '1:Sega',
            'opac.hold_notify' => 'phone',
            'opac.default_phone' => '987-654-3210',
            'opac.default_pickup_location' => '8',
            vendor_username => $VENDOR_USERNAME,
            vendor_password => $VENDOR_PASSWORD,
            security_token => generate_ecard_token($SHARED_SECRET),
        });
        ok($response->is_success, 'Registration submission succeeds (but not necessarily the Registration itself)');
        my $content = eval { decode_json($response->decoded_content) };
        ok(!$@, 'Response is valid JSON');
        diag(Dumper($content || (defined $response->title ? $response->title : $response)));
        $first_patron_id = $content->{patron_id};
    }

    my $expire_date;

    # open-ils.actor.user.opac.renewal
    {
        my $response = osrf_gateway_request($ua, [
            service => 'open-ils.actor',
            method => 'open-ils.actor.user.opac.renewal',
            param => encode_json($auth_token),
            param => encode_json($first_patron_id)
        ]);
        ok($response->is_success, 'Patron retrieval submission succeeds');
        my $content = eval { decode_json($response->decoded_content) };
        ok(!$@, 'Response is valid JSON');
        is($content->{status}, '200', 'Retrieval succeeded');
        my $user = $content->{payload}[0]->{user};
        is($user->{email}, $ecard_data{'email'}, 'Email matches');
        $expire_date = $user->{expire_date};
        diag("expire_date = $expire_date");
        ok(defined $user->{settings}->{'opac.hold_notify'}, 'opac.hold_notify is set');
        is($user->{settings}->{'opac.hold_notify'}, 'phone', 'opac.hold_notify is set to phone');
        ok(defined $user->{settings}->{'opac.default_phone'}, 'opac.default_phone is set');
        is($user->{settings}->{'opac.default_phone'}, '987-654-3210', 'opac.default_phone is set to 987-654-3210');
        ok(defined $user->{settings}->{'opac.default_pickup_location'}, 'opac.default_pickup_location is set');
        is($user->{settings}->{'opac.default_pickup_location'}, '8', 'opac.default_pickup_location is set to 8');
        diag('user settings => ' . Dumper($user->{settings}));
        ok(defined $user->{stat_cat_entries}->{1}, 'stat cat is set');
        is($user->{stat_cat_entries}->{1}, 'Sega', 'stat cat is set to Sega');
        diag('user stat cat entries => ' . Dumper($user->{stat_cat_entries}));
        #diag(Dumper($content || (defined $response->title ? $response->title : $response)));
    }

    # Test: Expired token
    {
        my $old_timestamp = (int(time()) - 3600*1000);  # 1 hour ago
        my $response = ecard_submit_request($ua, {
            %ecard_data,
            vendor_username => $VENDOR_USERNAME,
            vendor_password => $VENDOR_PASSWORD,
            security_token => generate_ecard_token($SHARED_SECRET, $old_timestamp),
        });
        ok(!$response->is_success, 'Registration submission with Expired token fails');
        diag(Dumper(defined $response->title ? $response->title : $response));
    }

    # Test: Future token
    {
        my $future_timestamp = (int(time()) + 3600*1000);  # 1 hour ahead
        my $response = ecard_submit_request($ua, {
            %ecard_data,
            vendor_username => $VENDOR_USERNAME,
            vendor_password => $VENDOR_PASSWORD,
            security_token => generate_ecard_token($SHARED_SECRET, $future_timestamp),
        });
        ok(!$response->is_success, 'Registration submission with Future token fails');
        diag(Dumper(defined $response->title ? $response->title : $response));
    }

    # Test: Very old token
    {
        my $very_old_timestamp = (int(time()) - 365 * 24 * 3600*1000);  # 1 year ago
        my $response = ecard_submit_request($ua, {
            %ecard_data,
            vendor_username => $VENDOR_USERNAME,
            vendor_password => $VENDOR_PASSWORD,
            security_token => generate_ecard_token($SHARED_SECRET, $very_old_timestamp),
        });
        ok(!$response->is_success, 'Registration submission with Very old token fails');
        diag(Dumper(defined $response->title ? $response->title : $response));
    }

    # Test: Very future token
    {
        my $very_future_timestamp = (int(time()) + 365 * 24 * 3600*1000);  # 1 year ahead
        my $response = ecard_submit_request($ua, {
            %ecard_data,
            vendor_username => $VENDOR_USERNAME,
            vendor_password => $VENDOR_PASSWORD,
            security_token => generate_ecard_token($SHARED_SECRET, $very_future_timestamp),
        });
        ok(!$response->is_success, 'Registration submission with Very future token fails');
        diag(Dumper(defined $response->title ? $response->title : $response));
    }

    # Test: API test mode
    {
        my $response = ecard_submit_request($ua, {
            testmode => "API",
            vendor_username => $VENDOR_USERNAME,
            vendor_password => $VENDOR_PASSWORD,
            security_token => generate_ecard_token($SHARED_SECRET),
        });
        ok($response->is_success, 'API test mode submission succeeds');
        my $content = eval { decode_json($response->decoded_content) };
        ok(!$@, 'API test mode response is valid JSON');
        #diag(Dumper($content || (defined $response->title ? $response->title : $response)));
    }

    # Test: Data mode test
    {
        my $response = ecard_submit_request($ua, {
            datamode => "all",
            vendor_username => $VENDOR_USERNAME,
            vendor_password => $VENDOR_PASSWORD,
            security_token => generate_ecard_token($SHARED_SECRET),
        });
        ok($response->is_success, 'Data mode test submission succeeds');
        my $content = eval { decode_json($response->decoded_content) };
        ok(!$@, 'Data mode response is valid JSON');
        #diag(Dumper($content || (defined $response->title ? $response->title : $response)));
    }

    # Test: Username taken
    {
        my $response = ecard_submit_request($ua, {
            %ecard_data,
            usrname => "99999360638",
            vendor_username => $VENDOR_USERNAME,
            vendor_password => $VENDOR_PASSWORD,
            security_token => generate_ecard_token($SHARED_SECRET),
        });
        ok($response->is_success, 'Registration submission succeeds');
        my $content = eval { decode_json($response->decoded_content) };
        ok(!$@, 'Response is valid JSON');
        is($content->{status}, 'USERNAME_TAKEN', 'Username is already taken');
        diag(Dumper($content || (defined $response->title ? $response->title : $response)));
    }

    my $patron_id;

    my $usrname = 'purple_pineapple_' . time();

    my $today_dt = DateTime->now(time_zone => 'local');
    my $today_iso = $today_dt->iso8601();
    my $today_plus_15_dt = DateTime->now(time_zone => 'local')->add(days => 15);
    my $today_plus_15_iso = $today_plus_15_dt->iso8601();
    my $today_plus_60_dt = DateTime->now(time_zone => 'local')->add(days => 60);
    my $today_plus_60_iso = $today_plus_60_dt->iso8601();
    my $today_minus_60_dt = DateTime->now(time_zone => 'local')->subtract(days => 60);
    my $today_minus_60_iso = $today_minus_60_dt->iso8601();

    # Test: User registered (already expired for subsequent renewal testing)
    {
        my $response = ecard_submit_request($ua, {
            %ecard_data,
            expire_date => $today_minus_60_iso, # This will override the default provided by the permission group
            usrname => $usrname,
            vendor_username => $VENDOR_USERNAME,
            vendor_password => $VENDOR_PASSWORD,
            security_token => generate_ecard_token($SHARED_SECRET),
        });
        ok($response->is_success, 'Registration submission succeeds');
        my $content = eval { decode_json($response->decoded_content) };
        ok(!$@, 'Response is valid JSON');
        is($content->{status}, 'OK', 'User registered');
        diag(Dumper($content || (defined $response->title ? $response->title : $response)));
        $patron_id = $content->{patron_id};
        diag("patron_id = $patron_id");
    }

    # open-ils.actor.user.opac.renewal
    {
        my $response = osrf_gateway_request($ua, [
            service => 'open-ils.actor',
            method => 'open-ils.actor.user.opac.renewal',
            param => encode_json($auth_token),
            param => encode_json($patron_id)
        ]);
        ok($response->is_success, 'Patron retrieval submission succeeds');
        my $content = eval { decode_json($response->decoded_content) };
        ok(!$@, 'Response is valid JSON');
        is($content->{status}, '200', 'Retrieval succeeded');
        my $user = $content->{payload}[0]->{user};
        is($user->{email}, $ecard_data{'email'}, 'Email matches');
        $expire_date = $user->{expire_date};
        diag("expire_date = $expire_date");
        #diag(Dumper($content || (defined $response->title ? $response->title : $response)));
    }

    my $formatter = DateTime::Format::ISO8601->new();
    my $dt_date = $formatter->parse_datetime($expire_date);
    $dt_date->add( years => 1 );
    my $new_expire_date = $dt_date->strftime('%Y-%m-%dT%H:%M:%S%z');

    # Test: User editing (for renewal)
    {
        my $response = ecard_submit_request($ua, {
            patron_id => $patron_id,
            %ecard_data,
            expire_date => $new_expire_date, # this will supplant the one coming from %ecard_data
            vendor_username => $VENDOR_USERNAME,
            vendor_password => $VENDOR_PASSWORD,
            security_token => generate_ecard_token($SHARED_SECRET),
        });
        ok($response->is_success, 'Renewal submission succeeds');
        my $content = eval { decode_json($response->decoded_content) };
        #diag(Dumper($content || (defined $response->title ? $response->title : $response)));
        ok(!$@, 'Response is valid JSON');
        is($content->{status}, undef, 'User not eligible for renewal'); # This is only true in this case because the user has not logged in immediately prior to the check
    }

    # open-ils.actor.user.opac.renewal
    {
        my $response = osrf_gateway_request($ua, [
            service => 'open-ils.actor',
            method => 'open-ils.actor.user.opac.renewal',
            param => encode_json($auth_token),
            param => encode_json($patron_id)
        ]);
        ok($response->is_success, 'Patron retrieval submission succeeds');
        my $content = eval { decode_json($response->decoded_content) };
        ok(!$@, 'Response is valid JSON');
        is($content->{status}, '200', 'Retrieval succeeded');
        my $user = $content->{payload}[0]->{user};
        is($user->{email}, $ecard_data{'email'}, 'Email matches');
        my $testing_expire_date = $user->{expire_date};
        isnt($testing_expire_date, $new_expire_date, 'Expire date did not change');
        diag("expire_date = $expire_date");
        #diag(Dumper($content || (defined $response->title ? $response->title : $response)));
    }

    my $patron_ua = LWP::UserAgent->new(
        cookie_jar => HTTP::Cookies->new,
        ssl_opts => {
            verify_hostname => 0,
            SSL_verify_mode => 0x00
        }
    );

    my $login_resp = $patron_ua->post(
        "https://$host/eg/opac/login", {
          username => $usrname,
          password => $ecard_data{'passwd'},
        }
    );

    # diag(Dumper($login_resp));
    ok($login_resp->is_redirect, 'Patron login is successful (redirects)');

    diag('starting second page load...');

    # This sets an eligibility key in a cache that gets checked
    my $second_page_load = $patron_ua->get("https://$host/eg/opac/myopac/prefs");
    # diag(Dumper($second_page_load));

    # Test: User editing (for renewal)
    {
        my $response = ecard_submit_request($ua, {
            patron_id => $patron_id,
            %ecard_data,
            expire_date => $new_expire_date, # this will supplant the one coming from %ecard_data
            vendor_username => $VENDOR_USERNAME,
            vendor_password => $VENDOR_PASSWORD,
            security_token => generate_ecard_token($SHARED_SECRET),
        });
        ok($response->is_success, 'Renewal submission succeeds');
        my $content = eval { decode_json($response->decoded_content) };
        diag(Dumper($content || (defined $response->title ? $response->title : $response)));
        ok(!$@, 'Response is valid JSON');
        isnt($content->{status}, undef, 'User should be eligible (status not undef)');
        isnt($content->{status}, 'UPDATE_ERR', 'User should be eligible (status not UPDATE_ERR)');
    }

    # open-ils.actor.user.opac.renewal
    {
        my $response = osrf_gateway_request($ua, [
            service => 'open-ils.actor',
            method => 'open-ils.actor.user.opac.renewal',
            param => encode_json($auth_token),
            param => encode_json($patron_id)
        ]);
        ok($response->is_success, 'Patron retrieval submission succeeds');
        my $content = eval { decode_json($response->decoded_content) };
        ok(!$@, 'Response is valid JSON');
        is($content->{status}, '200', 'Retrieval succeeded');
        my $user = $content->{payload}[0]->{user};
        is($user->{email}, $ecard_data{'email'}, 'Email matches');
        my $testing_expire_date = $user->{expire_date};
        is($testing_expire_date, $new_expire_date, 'Expire date did change');
        diag("expire_date = $expire_date");
        #diag(Dumper($content || (defined $response->title ? $response->title : $response)));
    }


    {
        my $response = ecard_submit_request($ua, {
            patron_id => $patron_id,
            %ecard_data,
            asceum => '1:Nintendo',
            'opac.hold_notify' => 'email',
            'opac.default_phone' => '555-555-5555',
            'opac.default_pickup_location' => '4',
            expire_date => $new_expire_date, # TODO: this will get reset if we don't provide it; I think that's a bug
            vendor_username => $VENDOR_USERNAME,
            vendor_password => $VENDOR_PASSWORD,
            security_token => generate_ecard_token($SHARED_SECRET),
        });
        ok($response->is_success, 'Renewal submission succeeds for adding stat cat and settings');
        my $content = eval { decode_json($response->decoded_content) };
        diag(Dumper($content || (defined $response->title ? $response->title : $response)));
        ok(!$@, 'Response is valid JSON');
        isnt($content->{status}, undef, 'User should be eligible (status not undef)');
        isnt($content->{status}, 'UPDATE_ERR', 'User should be eligible (status not UPDATE_ERR)');
    }

    # open-ils.actor.user.opac.renewal
    {
        my $response = osrf_gateway_request($ua, [
            service => 'open-ils.actor',
            method => 'open-ils.actor.user.opac.renewal',
            param => encode_json($auth_token),
            param => encode_json($patron_id)
        ]);
        ok($response->is_success, 'Patron retrieval submission for checking added stat cat and settings succeeds');
        my $content = eval { decode_json($response->decoded_content) };
        ok(!$@, 'Response is valid JSON');
        is($content->{status}, '200', 'Retrieval succeeded');
        my $user = $content->{payload}[0]->{user};
        is($user->{email}, $ecard_data{'email'}, 'Email matches');
        $expire_date = $user->{expire_date};
        diag("expire_date = $expire_date");
        ok(defined $user->{settings}->{'opac.hold_notify'}, 'opac.hold_notify is set');
        is($user->{settings}->{'opac.hold_notify'}, 'email', 'opac.hold_notify is set to email');
        ok(defined $user->{settings}->{'opac.default_phone'}, 'opac.default_phone is set');
        is($user->{settings}->{'opac.default_phone'}, '555-555-5555', 'opac.default_phone is set to 555-555-5555');
        ok(defined $user->{settings}->{'opac.default_pickup_location'}, 'opac.default_pickup_location is set');
        is($user->{settings}->{'opac.default_pickup_location'}, '4', 'opac.default_pickup_location is set to 4');
        diag('user settings => ' . Dumper($user->{settings}));
        ok(defined $user->{stat_cat_entries}->{1}, 'stat cat is set');
        is($user->{stat_cat_entries}->{1}, 'Nintendo', 'stat cat is set to Nintendo');
        diag('user stat cat entries => ' . Dumper($user->{stat_cat_entries}));
        #diag(Dumper($content || (defined $response->title ? $response->title : $response)));
    }

    # now for editing
    {
        my $response = ecard_submit_request($ua, {
            patron_id => $patron_id,
            %ecard_data,
            asceum => '1:Sega',
            'opac.hold_notify' => 'phone:email',
            'opac.default_phone' => '999-999-9999',
            'opac.default_pickup_location' => '5',
            expire_date => $new_expire_date, # TODO: this will get reset if we don't provide it; I think that's a bug
            vendor_username => $VENDOR_USERNAME,
            vendor_password => $VENDOR_PASSWORD,
            security_token => generate_ecard_token($SHARED_SECRET),
        });
        ok($response->is_success, 'Renewal submission succeeds for editing stat cats and settings');
        my $content = eval { decode_json($response->decoded_content) };
        diag(Dumper($content || (defined $response->title ? $response->title : $response)));
        ok(!$@, 'Response is valid JSON');
        isnt($content->{status}, undef, 'User should be eligible (status not undef)');
        isnt($content->{status}, 'UPDATE_ERR', 'User should be eligible (status not UPDATE_ERR)');
    }

    {
        my $response = osrf_gateway_request($ua, [
            service => 'open-ils.actor',
            method => 'open-ils.actor.user.opac.renewal',
            param => encode_json($auth_token),
            param => encode_json($patron_id)
        ]);
        ok($response->is_success, 'Patron retrieval submission for checking edited stat cat and settings succeeds');
        my $content = eval { decode_json($response->decoded_content) };
        ok(!$@, 'Response is valid JSON');
        is($content->{status}, '200', 'Retrieval succeeded');
        my $user = $content->{payload}[0]->{user};
        is($user->{email}, $ecard_data{'email'}, 'Email matches');
        $expire_date = $user->{expire_date};
        diag("expire_date = $expire_date");
        ok(defined $user->{settings}->{'opac.hold_notify'}, 'opac.hold_notify is set');
        is($user->{settings}->{'opac.hold_notify'}, 'phone:email', 'opac.hold_notify is set to phone:email');
        ok(defined $user->{settings}->{'opac.default_phone'}, 'opac.default_phone is set');
        is($user->{settings}->{'opac.default_phone'}, '999-999-9999', 'opac.default_phone is set to 999-999-9999');
        ok(defined $user->{settings}->{'opac.default_pickup_location'}, 'opac.default_pickup_location is set');
        is($user->{settings}->{'opac.default_pickup_location'}, '5', 'opac.default_pickup_location is set to 5');
        diag('user settings => ' . Dumper($user->{settings}));
        ok(defined $user->{stat_cat_entries}->{1}, 'stat cat is set');
        is($user->{stat_cat_entries}->{1}, 'Sega', 'stat cat is set to Sega');
        diag('user stat cat entries => ' . Dumper($user->{stat_cat_entries}));
        #diag(Dumper($content || (defined $response->title ? $response->title : $response)));
    }

    # this time, let's edit and leave the settings alone and see if they change (they should not)
    {
        my $response = ecard_submit_request($ua, {
            patron_id => $patron_id,
            %ecard_data,
            expire_date => $new_expire_date, # TODO: this will get reset if we don't provide it; I think that's a bug
            vendor_username => $VENDOR_USERNAME,
            vendor_password => $VENDOR_PASSWORD,
            security_token => generate_ecard_token($SHARED_SECRET),
        });
        ok($response->is_success, 'Renewal submission succeeds for editing stat cats and settings');
        my $content = eval { decode_json($response->decoded_content) };
        diag(Dumper($content || (defined $response->title ? $response->title : $response)));
        ok(!$@, 'Response is valid JSON');
        isnt($content->{status}, undef, 'User should be eligible (status not undef)');
        isnt($content->{status}, 'UPDATE_ERR', 'User should be eligible (status not UPDATE_ERR)');
    }

    {
        my $response = osrf_gateway_request($ua, [
            service => 'open-ils.actor',
            method => 'open-ils.actor.user.opac.renewal',
            param => encode_json($auth_token),
            param => encode_json($patron_id)
        ]);
        ok($response->is_success, 'Patron retrieval submission for checking stat cats and settings');
        my $content = eval { decode_json($response->decoded_content) };
        ok(!$@, 'Response is valid JSON');
        is($content->{status}, '200', 'Retrieval succeeded');
        my $user = $content->{payload}[0]->{user};
        is($user->{email}, $ecard_data{'email'}, 'Email matches');
        $expire_date = $user->{expire_date};
        diag("expire_date = $expire_date");
        ok(defined $user->{settings}->{'opac.hold_notify'}, 'opac.hold_notify is set');
        is($user->{settings}->{'opac.hold_notify'}, 'phone:email', 'opac.hold_notify is set to phone:email');
        ok(defined $user->{settings}->{'opac.default_phone'}, 'opac.default_phone is set');
        is($user->{settings}->{'opac.default_phone'}, '999-999-9999', 'opac.default_phone is set to 999-999-9999');
        ok(defined $user->{settings}->{'opac.default_pickup_location'}, 'opac.default_pickup_location is set');
        is($user->{settings}->{'opac.default_pickup_location'}, '5', 'opac.default_pickup_location is set to 5');
        diag('user settings => ' . Dumper($user->{settings}));
        ok(defined $user->{stat_cat_entries}->{1}, 'stat cat is set');
        is($user->{stat_cat_entries}->{1}, 'Sega', 'stat cat is set to Sega');
        diag('user stat cat entries => ' . Dumper($user->{stat_cat_entries}));
        #diag(Dumper($content || (defined $response->title ? $response->title : $response)));
    }
};

done_testing();
